import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Tent, Bed, UtensilsCrossed, Compass, Calendar, CheckCircle, Flame, Map, Camera, Mountain } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Rent.css'

const Rent = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [returnDate, setReturnDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'tents', 
      name: 'Палатки', 
      icon: Tent,
      color: '#2D5016',
      description: 'Палатки различных размеров для комфортного отдыха на природе. От 2 до 8 мест.',
      items: ['Палатка 2-местная', 'Палатка 4-местная', 'Палатка 6-местная', 'Палатка 8-местная']
    },
    { 
      id: 'sleeping', 
      name: 'Спальные системы', 
      icon: Bed,
      color: '#8B4513',
      description: 'Спальники, коврики и подушки для комфортного сна в походе.',
      items: ['Спальник -10°C', 'Спальник 0°C', 'Спальник +10°C', 'Коврик самонадувающийся']
    },
    { 
      id: 'kitchen', 
      name: 'Кухня', 
      icon: UtensilsCrossed,
      color: '#FF6B35',
      description: 'Посуда, горелки, котелки и другое оборудование для приготовления пищи.',
      items: ['Газовая горелка', 'Котелок 2л', 'Набор посуды', 'Холодильник переносной']
    },
    { 
      id: 'navigation', 
      name: 'Навигация', 
      icon: Compass,
      color: '#4A90E2',
      description: 'Компасы, GPS-навигаторы и карты для ориентирования на местности.',
      items: ['GPS-навигатор', 'Компас', 'Карты местности', 'Рация']
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedCategory || !returnDate) {
      alert('Пожалуйста, выберите категорию и дату возврата')
      return
    }

    const saved = localStorage.getItem('campingRentals')
    const rentals = saved ? JSON.parse(saved) : []
    
    const categoryData = categories.find(c => c.id === selectedCategory)
    const randomItem = categoryData.items[Math.floor(Math.random() * categoryData.items.length)]
    
    const today = new Date()
    const startDate = today.toISOString().split('T')[0]
    
    const newRental = {
      id: Date.now(),
      item: randomItem,
      category: selectedCategory,
      startDate: startDate,
      returnDate: returnDate,
      status: 'active',
      review: null
    }
    
    rentals.push(newRental)
    localStorage.setItem('campingRentals', JSON.stringify(rentals))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="rent-page-nature">
      <div className="container">
        <section className="rent-header-nature">
          <div className="header-badge-nature">
            <span>АРЕНДА ОБОРУДОВАНИЯ</span>
          </div>
          <h1 className="rent-title-nature">
            Арендовать <span className="title-accent-nature">оборудование</span>
          </h1>
          <p className="rent-description-nature">
            Выберите категорию оборудования и укажите дату возврата. Все оборудование проверено и готово к использованию.
          </p>
        </section>

        <section className="rent-form-section-nature">
          <form className="form-nature" onSubmit={handleSubmit}>
            <div className="categories-selection-nature">
              {categories.map((category) => {
                const Icon = category.icon
                return (
                  <button
                    key={category.id}
                    type="button"
                    className={`category-card-nature ${selectedCategory === category.id ? 'selected' : ''}`}
                    onClick={() => setSelectedCategory(category.id)}
                    style={{ '--category-color': category.color }}
                  >
                    <div className="category-icon-wrapper-nature">
                      <Icon size={40} />
                    </div>
                    <h3 className="category-name-nature">{category.name}</h3>
                    <p className="category-desc-nature">{category.description}</p>
                    {selectedCategory === category.id && (
                      <div className="check-badge-nature">
                        <CheckCircle size={24} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedCategory && (
              <div className="date-section-nature">
                <div className="date-label-nature">
                  <Calendar size={20} />
                  <span>Дата возврата оборудования</span>
                </div>
                <input
                  type="date"
                  className="date-input-nature"
                  value={returnDate}
                  onChange={(e) => setReturnDate(e.target.value)}
                  min={minDateStr}
                  max={maxDateStr}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-nature" 
              disabled={!selectedCategory || !returnDate || isSubmitted}
            >
              <span>Оформить аренду</span>
            </button>

            {isSubmitted && (
              <div className="success-message-nature">
                <CheckCircle size={32} />
                <span>Аренда успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <PopularDestinationsSection />
      </div>
    </div>
  )
}

const PopularDestinationsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const destinations = [
    {
      icon: Mountain,
      title: 'Горные маршруты',
      description: 'Покорите вершины с нашим профессиональным снаряжением',
      gradient: 'linear-gradient(135deg, #2D5016 0%, #4A7C2A 100%)'
    },
    {
      icon: Map,
      title: 'Лесные тропы',
      description: 'Исследуйте тайгу с комфортом и безопасностью',
      gradient: 'linear-gradient(135deg, #8B4513 0%, #A0522D 100%)'
    },
    {
      icon: Camera,
      title: 'Фото-туры',
      description: 'Запечатлейте красоту природы с лучшим оборудованием',
      gradient: 'linear-gradient(135deg, #FF6B35 0%, #FF8C5A 100%)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`destinations-section-nature ${isVisible ? 'visible' : ''}`}
    >
      <div className="destinations-header-nature">
        <div className="destinations-badge-nature">
          <Flame size={20} />
          <span>ПОПУЛЯРНЫЕ НАПРАВЛЕНИЯ</span>
        </div>
        <h2 className="destinations-title-nature">Куда отправиться?</h2>
        <p className="destinations-description-nature">
          Выберите направление для вашего следующего приключения
        </p>
      </div>
      <div className="destinations-grid-nature">
        {destinations.map((dest, index) => {
          const Icon = dest.icon
          return (
            <div 
              key={index} 
              className="destination-card-nature"
              style={{ 
                animationDelay: `${index * 0.15}s`,
                background: dest.gradient
              }}
            >
              <div className="destination-icon-wrapper-nature">
                <Icon size={40} />
              </div>
              <h3 className="destination-title-nature">{dest.title}</h3>
              <p className="destination-text-nature">{dest.description}</p>
            </div>
          )
        })}
      </div>
    </section>
  )
}

export default Rent

